import bpy
import sys
import os

bpy.ops.wm.read_factory_settings(use_empty=True)

argv = sys.argv
glb_path = argv[argv.index("--") + 1]
fbx_path = argv[argv.index("--") + 2]

print(f"Converting: {glb_path} -> {fbx_path}")


bpy.ops.import_scene.gltf(filepath=glb_path)
 

bpy.ops.object.select_all(action='DESELECT')

mesh_object_name = "geometry_0"
mesh_obj = bpy.data.objects.get(mesh_object_name)

if mesh_obj and mesh_obj.type == 'MESH':
    print(f"Found mesh object: '{mesh_obj.name}'")

    
    if mesh_obj.parent:
        print(f"'{mesh_obj.name}' is parented to '{mesh_obj.parent.name}'. Unparenting.")
        original_matrix_world = mesh_obj.matrix_world.copy() 
        mesh_obj.parent = None                               
        mesh_obj.matrix_world = original_matrix_world        
        print(f"Unparented '{mesh_obj.name}' and restored its world transform.")

    
    mesh_obj.select_set(True)
    bpy.context.view_layer.objects.active = mesh_obj
 
    objects_to_delete = [obj for obj in bpy.context.scene.objects if obj.name != mesh_object_name]
    if objects_to_delete:
        bpy.ops.object.select_all(action='DESELECT')
        for obj_del in objects_to_delete:
            obj_del.select_set(True)
        if bpy.context.selected_objects: 
            print(f"Deleting other objects: {[o.name for o in bpy.context.selected_objects]}")
            bpy.ops.object.delete(use_global=False, confirm=False)
        
        bpy.ops.object.select_all(action='DESELECT')
        mesh_obj.select_set(True)
        bpy.context.view_layer.objects.active = mesh_obj

else:
    print(f"ERROR: Mesh object named '{mesh_object_name}' not found or is not a MESH. Skipping unparenting.")
    
    bpy.ops.object.select_all(action='DESELECT')
    for obj in bpy.context.scene.objects:
        if obj.type == 'MESH':
            obj.select_set(True)
            if not bpy.context.view_layer.objects.active: 
                 bpy.context.view_layer.objects.active = obj
    if not bpy.context.selected_objects:
        print("ERROR: No mesh objects found to export after failing to find 'geometry_0'.")
        sys.exit(1) 
 




if bpy.context.active_object and bpy.context.active_object.mode == 'EDIT':
    bpy.ops.object.mode_set(mode='OBJECT')


for obj in bpy.context.scene.objects:
    if obj.type == 'MESH':
        obj.select_set(True)
        bpy.context.view_layer.objects.active = obj 
    else:
        obj.select_set(False)

if bpy.context.selected_objects:
    bpy.ops.object.mode_set(mode='EDIT')
    bpy.ops.mesh.select_all(action='SELECT')
    bpy.ops.mesh.remove_doubles(threshold=0.0001, use_unselected=False)
    bpy.ops.mesh.select_all(action='DESELECT') 
    bpy.ops.object.mode_set(mode='OBJECT')
else:
    print("No mesh objects found or selected to perform merge by distance.")
 


fbx_dir = os.path.dirname(fbx_path)
textures_dir = os.path.join(fbx_dir, "textures")
os.makedirs(textures_dir, exist_ok=True)


for img in bpy.data.images:
    print(f"Found image: {img.name}, has_data: {img.has_data}, packed: {img.packed_file is not None}")
    
    if img.name == "Image_0" or img.has_data:  
        
        texture_path = os.path.join(textures_dir, f"{img.name}.png")
        
        
        img.filepath_raw = texture_path
        img.file_format = 'PNG'
        img.save()
        
        print(f"Saved {img.name} to {texture_path}")
        
        
        img.source = 'FILE'
        img.filepath = texture_path


for mat in bpy.data.materials:
    if mat.name == "material_0" or mat.use_nodes:  
        print(f"Processing material: {mat.name}")
        
        if mat.use_nodes:
            for node in mat.node_tree.nodes:
                if node.type == 'TEX_IMAGE' and node.image:
                    print(f"  Texture node using: {node.image.name}")
                    
                    if node.image.name == "Image_0":
                        node.image = bpy.data.images["Image_0"]


blend_path = os.path.join(fbx_dir, "temp.blend")
bpy.ops.wm.save_as_mainfile(filepath=blend_path)


bpy.ops.export_scene.fbx(
    filepath=fbx_path,
    use_selection=False,
    object_types={'MESH', 'EMPTY'},
    path_mode='RELATIVE',  
    embed_textures=True,  
    axis_forward='Y',
    axis_up='Z',
    mesh_smooth_type='FACE'
)


if os.path.exists(blend_path):
    os.remove(blend_path)
    
print(f"FBX exported to: {fbx_path}")
print(f"Textures in: {textures_dir}")


if os.path.exists(textures_dir):
    files = os.listdir(textures_dir)
    print(f"Texture files: {files}")
